// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.UI.Controls.DataModels.js
//
/// <reference path="../jquery-1.4.3.js" />
/// <reference path="TFS.Diag.debug.js" />

TFS.module("TFS.UI.Controls.Data", ["TFS.Core", "TFS.Core.Utils", "TFS.Diag", "TFS.UI"], function () {

    var Diag = TFS.Diag,
        Core = TFS.Core,
        domElem = TFS.UI.domElem,
        delegate = TFS.Core.delegate,
        ChecklistDataAdapter,           // For jsLint
        HierarchicalGridDataAdapter;    // For jsLint

    ///
    ///
    /// FieldDataProvider - Provides access to a (CSS) tree-structure payload with data caching
    ///                     to support quick lookup by id and path.
    ///                     The data provider can also manage a flat list of data
    ///

    function FieldDataProvider(nodes, options) {
        /// <summary>Populates the provider with the given items (nodes).</summary>
        /// <param name="nodes" type="Object">A collection of nodes in the following format:
        ///
        ///    Every node of the tree has the following format:
        ///    {
        ///         id:       <unique id, string, required>
        ///         parentId: <parent id, string, required (may be null)>
        ///         text:     <text for the node>
        ///         values:   <node values, array, required>
        ///         children: <array of nodes, node, optional>
        ///    }
        ///
        ///    Here is a sample declaration of grid items:
        ///
        ///    gridItems: [{
        ///        id: 0,
        ///        values: ["Root 1", "red", 100],
        ///        children: [{
        ///            id: 1,
        ///            values: ["Node 1-2", "green", 10],
        ///            children: [{
        ///                id: 2,
        ///                values: ["Leaf 1-2-1", "yellow", 70]
        ///            },
        ///            {
        ///                id: 3,
        ///                values: ["Leaf 1-2-2", "blue", 30]
        ///            }]
        ///        },
        ///        {
        ///            id: 4,
        ///            values: ["Root 2", "white", 50]
        ///        }]
        ///
        ///        "checked" is an array of tree item ids that must be initially checked in the grid. 
        ///        If this parameter is not provided nothing is checked.
        ///
        /// </param>
        /// <param name="options" type="object">
        /// OPTIONAL: Object with the following structure:
        ///   {
        ///     allowEmpty: <boolean: Indicates if empty values should be treated as valid or not.>
        ///     sort: comparison function for nodes if sorting is required
        ///   }
        /// </param>

        Diag.assertParamIsNotNull(nodes, "nodes");

        this._nodes = nodes;
        this._options = $.extend({
            allowEmpty: false,
            sort: null
        }, options);
        this._isTree = nodes.length > 0 && typeof (nodes[0]) === "object";
        this._events = new Sys.EventHandlerList();
        if (this._isTree) {
        this._sortChildren({ children: this._nodes }, true);
        }
        this._populateNodeMappings();
    }

    FieldDataProvider.extend({
        TREE_PATH_SEPERATER_CHAR: "\\",
        EVENT_NEW_ITEM: "new-item-event",
        EVENT_REMOVED_ITEM: "removed-item-event",
        EVENT_UPDATE_ITEM: "updated-item-event"
    });

    FieldDataProvider.prototype = {
        _events: null,
        _nodes: null,               // nodes for tree/list
        _idToNodeMap: null,         // optional mapping of node ID to the node.
        _pathToNodeMap: null,       // mapping of paths to the associated node.
        _isTree: false,             // flag of whether nodes are tree or list structure
        _options: null,             // options for the data provider.

        _populateNodeMappings: function () {
            /// <summary>Populate the mapping of path to associated node and id to node.</summary>
            var i, l,
                nodes = this._nodes;

            // Initialize the mapping to an empty object.
            this._pathToNodeMap = {};
            this._idToNodeMap = {};

            // Add each of the tree nodes to the mapping.
            if (this._isTree) {
                for (i = 0, l = nodes.length; i < l; i += 1) {
                    this._addNode(null, nodes[i]);
                }
            }
            else {
                for (i = 0, l = nodes.length; i < l; i += 1) {
                    this._pathToNodeMap[nodes[i]] = nodes[i];
                }
            }
        },

        reparentNode: function (node, newParent) {
            /// <summary>Move the node to a new parent.</summary>
            /// <param name="node" type="object">Node to be re-parented.</param>
            /// <param name="newParent" type="object">The new parent for the node.</param>

            Diag.assertParamIsObject(node, "node");
            Diag.assertParamIsObject(newParent, "newParent");

            // Remove the node from the current parent.
            this.removeNode(node.id);

            // Add the node to the new parent.
            this.addNode(node, newParent);
        },

        isValidValue: function (value) {
            /// <summary>Return true if the value is valid</summary>
            /// <param name="value" type="String">The value to check</param>
            /// <returns type="boolean" />
            Diag.assertParamIsString(value, "value");

            // If the value is empty and empty values are allowed, return true.
            if (value === "" && this._options.allowEmpty) {
                return true;
            }

            if (this._nodes.length === 0) {
                return value !== "";
            }

            // Cleanup any trailing \'s.
            value = this._cleanupPath(value);

            return this._pathToNodeMap.hasOwnProperty(value);
        },

        isTree: function () {
            /// <summary>return true if the data represented is tree</summary>
            /// <returns type="Boolean" />

            return this._isTree;
        },

        getNodes: function () {
            /// <summary>get Nodes to use in the combo box</summary>
            /// <returns type="object" />

            return this._nodes;
        },

        getNode: function (nodeText) {
            /// <summary>Get a node by its text</summary>
            /// <param name="nodeText" type="String">text of the node to lookup</param>
            /// <returns type="String or object" />
            Diag.assertParamIsString(nodeText, "nodeText");

            // Trim off any trailing \'s.
            nodeText = this._cleanupPath(nodeText);

            return this._pathToNodeMap[nodeText] || null;
        },

        getNodeFromId: function (nodeId) {
            /// <summary>Get the node associated with the id provided.</summary>
            /// <param name="nodeId" type="string">id of the node</param>
            /// <returns type="String or object" />

            Diag.assertParamIsString(nodeId, "nodeId");

            return this._idToNodeMap[nodeId] || null;
        },

        updateNode: function (node) {
            /// <summary>Update node in the tree.</summary>
            /// <param name="node" type="object">Node to update.</param>
            /// <returns type="Object">The updated node data</returns>
            Diag.assertParamIsObject(node, "node");

            // check that the incoming node
            //  a) exists
            //  b) the parent id hasn't changed
            var currentNode = this.getNodeFromId(node.id);

            // check node exists
            if (typeof currentNode === 'undefined') {
                Diag.fail("Expected to find the node in the tree when updating a node. ID: " + node.id);
                return null;
            }

            // check node's parent matches incoming node's parent
            if (currentNode.parentId !== node.parentId) {
                Diag.fail("Expected the parentId of the node to be the same. If you're re-parenting, remove the node, then re-add it. ID: " + node.id);
                return null;
            }

            if (currentNode.text !== node.text) {

                // When the text changes, we need to reconstruct the internal indexes

                // Remove the node from all internal indexes.
                this._clearCache(currentNode);

                // text has changed, which affects the path and the internal maps.
                currentNode.text = node.text;

                // Re-index the node.
                this._addNode(currentNode.parent, currentNode);
            }

            // Update the values for the node and ensure the node remains sorted
            currentNode.values = node.values.slice(0);

            // If there is a parent, sort the children.
            if (currentNode.parent) {
                this._sortChildren(currentNode.parent);
            }
            
            this._raiseUpdateItem({
                node: currentNode
            });

            return currentNode;
        },

        _sortChildren: function (node, recursive, sort) {
            /// <summary>Sort the children of a node (possibly recursively)</summary>
            /// <param name="node" type="object">The node whose children will be sorted</param>
            /// <param name="recursive" type="Boolean">(optional)If true, then the sort will proceed recursively through descendents</param>
            /// <param name="sort" type="Function">(optional) Comparison function for sorting the nodes. 
            ///     If not supplied, the sort function from the options will be used.</param>
            Diag.assertParamIsObject(node, "node");

            var nodes = node.children || [],
                nodeCount = nodes.length;

            sort = sort || this._options.sort;

            if ($.isFunction(sort)) {
                nodes.sort(sort);

                if (recursive) {
                    while (nodeCount > 0) {
                        nodeCount -= 1;
                        this._sortChildren(nodes[nodeCount], true, sort);
                    }
                }
            }
        },

        getRootNode: function () {
            /// <summary>Gets the first root node of the payload.</summary>

            var nodes = this.getNodes(),
                rootNode = null;

            if (nodes && nodes.length > 0) {
                rootNode = nodes[0];
            }

            return rootNode;
        },

        getPreviousSiblingNode: function (id) {
            /// <summary>Get the previous sibling node of the node identified by "id"</summary>
            /// <param name="id" type="String">The id (Guid string) for the node</param>
            Diag.assertParamIsString(id, "id");

            var i, l,
                prev = null,
                node = this.getNodeFromId(id),
                nodes;

            if (node) {
                nodes = node.parent ? node.parent.children : this._nodes;

                for (i = 0, l = nodes.length; i < l; i += 1) {
                    if (nodes[i] === node) {
                        break;
                    }
                    prev = nodes[i];
                }
            }

            return prev;
        },

        removeNode: function (id) {
            /// <summary>Deletes the specified node from the source list and all cached indexes.
            /// Returns the removed node</summary>
            /// <param name="id">The ID of the node in which to remove.</param>
            /// <returns type="object" />

            Diag.assertParamIsString(id, "id");

            var node = this.getNodeFromId(id),
                treeSize;
            
            if (node) {
                treeSize = this._getChildrenCount(node) + 1;

                if (node.parent && node.parent.children) {
                    node.parent.children = node.parent.children.subtract([node]);
                }

                this._clearCache(node);
            
                this._raiseRemovedItem({
                    node: node,
                    parent: node.parent,
                    treeSize: treeSize
                });
            }

            return node;
        },

        addNode: function (node, parent) {
            /// <summary>Add the provided node to the tree.</summary>
            /// <param name="node" type="object">New node to add.</param>
            /// <param name="parent" type="object">The node to parent under</param>

            Diag.assertParamIsObject(node, "node");
            Diag.assertParamIsObject(parent, "parent");

            var children = parent.children;

            // Check that the node is not already part of the tree
            if (this.getNodeFromId(node.id)) {
                Diag.fail("Shouldn't be adding a node to the tree that already exists. Id: " + node.id);
                return null;
            }

            // Check that the parent node in the tree
            if (!this.getNodeFromId(parent.id)) {
                Diag.fail(String.format("Couldn't find parent node in the tree when adding a child node. Parent Id: {0}; Child Id: {1}.", parent.id, node.id));
                return null;
            }

            // add into sorted position within the children && sort the node's children
            children.push(node);
            node.parentId = parent.id;
            this._sortChildren(parent, false);  // false = don't recurse, just ensure the new node is in the correct position
            this._sortChildren(node, true);     // true = ensure the node's children are sorted (recursively)

            // Add the new node to the mappings.
            this._addNode(parent, node);

            this._raiseNewItem({
                node: node,
                parent: parent
            });

            return node;
        },

        cloneSource: function () {
            /// <summary>Returns a clone, or deep-copy, of the source collection.</summary>
            return [HierarchicalGridDataAdapter.cloneNode(this.getRootNode())];
        },

        _addNode: function (parent, node) {
            /// <summary>Adds the specified node to all cached indexes.</summary>
            /// <param name="node">The node in which to add.</param>
            /// <param name="parent">The parent of the node in which to add.</param>

            var i, l,
                path = (parent === null ? "" : parent.path + FieldDataProvider.TREE_PATH_SEPERATER_CHAR) + node.text;

            // Save the path to this node.
            node.path = path;

            // Record the parent for upwards traversal
            if (parent) {
                Diag.assert(parent.id === node.parentId, String.format("Expected the parent's id ({0}) to match the node's parentId ({1})", parent.id, node.parentId));
            }
            node.parent = parent;

            // Add an entry in path to node and id to node mappings for this node.
            this._pathToNodeMap[path] = node;
            this._idToNodeMap[node.id] = node;

            // If the node has children, add them.
            if (node.children) {
                for (i = 0, l = node.children.length; i < l; i += 1) {
                    this._addNode(node, node.children[i]);
                }
            }
        },

        _clearCache: function (node) {
            var i, l;

            if (node) {

                delete this._pathToNodeMap[node.path];
                delete this._idToNodeMap[node.id];

                if (node.children) {
                    for (i = 0, l = node.children.length; i < l; i += 1) {
                        this._clearCache(node.children[i]);
                    }
                }
            }
        },

        _cleanupPath: function (path) {
            /// <summary>Cleans up the path removing any trailing \'s</summary>
            /// <param name="path" type="String">Path to be cleaned up.</param>
            Diag.assertParamIsString(path, "path");

            var result = path;

            // If this is a tree, trim off any trailing \'s.
            if (result.length > 0 && this.isTree() && result.charAt(result.length - 1) === FieldDataProvider.TREE_PATH_SEPERATER_CHAR) {
                result = result.substr(0, result.length - 1);
            }

            return result;
        },
        
        _getChildrenCount: function (node) {
            /// <summary>Gets a count of all the specified nodes' children, recursively.</summary>
            /// <param name="node" type="Object">The node whose children to count.</param>
            /// <remarks>The children collection of node is optional so we only need to traverse them if present.</remarks>
            
            Diag.assertParamIsObject(node, "node");
            
            var i, l,
                childrenCount = 0;
            
            if ($.isArray(node.children)) {
                childrenCount = node.children.length;
                
                for (i = 0, l = childrenCount; i < l; i += 1) {
                    childrenCount += this._getChildrenCount(node.children[i]);
                }
            }
            
            return childrenCount;
        },

        _raiseNewItem: function (args) {
            /// <summary>Notifies listeners of NewItem</summary>
            /// <param name="args" type="object">args</param>

            Diag.assertParamIsObject(args, "args");

            var handler = this._events.getHandler(FieldDataProvider.EVENT_NEW_ITEM);

            if (handler) {
                handler(this, args);
            }
        },

        attachNewItem: function (handler) {
            /// <summary> Attach a handler for the EVENT_NEW_ITEM event. </summary>       
            /// <param name="handler" type="Function">The handler to attach</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.addHandler(FieldDataProvider.EVENT_NEW_ITEM, handler);
        },

        detachNewItem: function (handler) {
            /// <summary>Remove a handler for the EVENT_NEW_ITEM event</summary>
            /// <param name="handler" type="Function">The handler to remove</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.removeHandler(FieldDataProvider.EVENT_NEW_ITEM, handler);
        },

        _raiseRemovedItem: function (args) {
            /// <summary>Notifies listeners of that a work item was removed.</summary>
            /// <param name="args" type="object">args</param>

            Diag.assertParamIsObject(args, "args");

            var handler = this._events.getHandler(FieldDataProvider.EVENT_REMOVED_ITEM);

            if (handler) {
                handler(this, args);
            }
        },

        attachRemovedItem: function (handler) {
            /// <summary>Attach a handler for the removed item event. </summary>       
            /// <param name="handler" type="Function">
            /// The handler to attach.  This will be invoked with an argument in the following format:
            ///   {
            ///     workItemIndex: index,
            ///     treeSize: treeSize
            ///   }
            /// </param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.addHandler(FieldDataProvider.EVENT_REMOVED_ITEM, handler);
        },

        detachRemovedItem: function (handler) {
            /// <summary>Remove a handler for the removed item event.</summary>
            /// <param name="handler" type="Function">The handler to remove</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.removeHandler(FieldDataProvider.EVENT_REMOVED_ITEM, handler);
        },
        
        _raiseUpdateItem: function (args) {
            /// <summary>Notifies listeners of updateItem</summary>
            /// <param name="args" type="object">args</param>

            Diag.assertParamIsObject(args, "args");

            var handler = this._events.getHandler(FieldDataProvider.EVENT_UPDATE_ITEM);

            if (handler) {
                handler(this, args);
            }
        },

        attachUpdateItem: function (handler) {
            /// <summary> Attach a handler for the EVENT_UPDATE_ITEM event. </summary>       
            /// <param name="handler" type="Function">The handler to attach</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.addHandler(FieldDataProvider.EVENT_UPDATE_ITEM, handler);
        },

        detachUpdateItem: function (handler) {
            /// <summary>Remove a handler for the EVENT_UPDATE_ITEM event</summary>
            /// <param name="handler" type="Function">The handler to remove</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.removeHandler(FieldDataProvider.EVENT_UPDATE_ITEM, handler);
        }
    };


    ///
    ///
    /// HierarchicalGridDataAdapter - Binds a grid control to a field data provider in a way 
    ///                               such that the grid can understand how do render the hierarchy.
    ///

    HierarchicalGridDataAdapter = function (fieldDataProvider, grid, options) {
        /// <summary>Creates an adapter to provide data from a field data provider to a grid control.</summary>
        /// <param name="fieldDataProvider" type="object">The field data provider that represents a tree graph of data.</param>
        /// <param name="grid" type="object">The grid control in which to bind to the data provider.</param>
        /// <param name="options" type="object">Options that may be used to customize the behavior of this provider.</param>

        Diag.assertParamIsObject(fieldDataProvider, "fieldDataProvider");
        Diag.assertParamIsObject(grid, "grid");

        this._grid = grid;
        this._options = $.extend({}, options);
        this.dataProvider = fieldDataProvider;

        this.dataProvider.attachNewItem(delegate(this, function (source, args) {
            this._addNewItem(args.node, args.parent);
        }));

        this.dataProvider.attachRemovedItem(delegate(this, function (source, args) {
            this._removeItem(args.node, args.parent, args.treeSize);
        }));
        
        this.dataProvider.attachUpdateItem(delegate(this, function (source, args) {
            this._updateItem(args.node);
        }));
    };

    HierarchicalGridDataAdapter.extend({
        _ITEM_ID_DATA_SOURCE_INDEX: -1,

        bind: function (fieldDataProvider, grid, options) {
            /// <summary>Binds a field data provider to a grid control.</summary>

            Diag.assertParamIsObject(fieldDataProvider, "fieldDataProvider");
            Diag.assertParamIsObject(grid, "grid");

            var i, l,
                columns,
                gridAdapter = new this(fieldDataProvider, grid, options);

            columns = grid._options.columns;
            
            for (i = 0, l = columns.length; i < l; i++) {
                delete columns[i].index;
            }

            gridAdapter.refresh();

            return gridAdapter;
        },

        cloneNode: function (node) {
            /// <summary>Clones the specified node and all its children, returning the cloned node.</summary>
            /// <param name="node">The node to clone.</param>

            var newNode = $.extend({}, node),
            i, l;

            newNode.children = [];
            delete newNode.parent;

            for (i = 0, l = node.children.length; i < l; i += 1) {
                newNode.children[i] = HierarchicalGridDataAdapter.cloneNode(node.children[i]);
            }

            return newNode;
        }
    });

    HierarchicalGridDataAdapter.prototype = {
        _flattenedItems: null,
        _grid: null,
        _expandStates: [],
        _options: {},
        _expandStatesManager: null,
        dataProvider: null,

        getGrid: function () {
            /// <summary>Gets the grid that this adapter is associated with</summary>
            /// <returns type="Object" />
            return this._grid;
        },
        
        refresh: function (calculateOnly) {
            /// <summary>Refreshes the contents of the grid with the current contents of the field data provider.</summary>
            /// <param name="calculateOnly" type="Boolean">Indicates whether the refresh should update the bound grids' data 
            /// source and expand states or should just rebuild all internal indexes.  When <i>true</i>, this function will
            /// only rebuild the internal indexes and caches without updating the bound grid.  This is sometimes useful when 
            /// you need to recalculate indexes during a reparent but don't want to update the grid until the reparent has
            /// completed.</param>

            var nodes = this.dataProvider.getNodes(),
                backupExpandStates = [],
                expandStates = this._grid.getExpandStates(),
                newExpandStates = [];

            if (expandStates) {
                backupExpandStates = expandStates.slice(0);
            }
                    
            this._flattenedItems = [];

            // Reset the list of root nodes
            this._createDataSource(nodes, this._flattenedItems, newExpandStates, 1);

            if (!calculateOnly) {
                // We don't want to update the expanded states unless we intend on updating the
                // grid.
                this._expandStates = newExpandStates;
                
                // if we are refreshing on the same view, then keep the expanded states as is
                // if we are refreshing to a different view then the expanded states lengths won't match

                if (this._expandStates && (this._expandStates.length === backupExpandStates.length)) {
                    this._expandStates = backupExpandStates;
                }

                // Bind the datasource to the grid
                this._grid.setDataSource(this._flattenedItems, this._expandStates, this._grid._options.columns);
            }
        },

        getNodeForDataIndex: function (dataIndex) {
            /// <summary>Gets the node associated with the data index.</summary>
            /// <param name="dataIndex" type="number">Data index of the node to lookup.</param>

            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            var item = this._flattenedItems[dataIndex];

            if (item) {
                return this.dataProvider.getNodeFromId(item[HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX]);
            }
            else {
                return null;
            }
        },

        getParentNodeIndexForDataIndex: function (dataIndex) {
            /// <summary>Gets the parent of the node associated with the data index.</summary>
            /// <param name="dataIndex" type="number">Data index of the node to lookup.</param>
            /// <returns>A grid row index for the parent node of the node in the specified dataIndex of the grid.</returns>
            
            // If we don't have a child data Index to start off with, don't
            // try to find the parent index because there won't be one.
            // Note, if we have a)a zero value, b)a null value, or c) an undefined value for dataIndex
            // don't proceed since we will not have a parent to fetch here.
            if (!dataIndex) {
                return -1;
            }
            
            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            var node = this.getNodeForDataIndex(dataIndex),
            parentIndex = -1;

            if (node) {
                parentIndex = this.getDataIndexFromNode(node.parent);
            }
            
            return parentIndex;
        },

        getDataIndexFromNode: function (node) {
            /// <summary>Gets the data index for the specified node ID.</summary>
            /// <param name="node" type="Object">The node whose data index is to be retrieved.</param>
            
            Diag.assertParamIsObject(node, "node");
            return node.dataIndex;
        },

        cloneSource: function () {
            /// <summary>Returns a clone, or deep-copy, of the source collection.</summary>

            return [HierarchicalGridDataAdapter.cloneNode(this.fieldDataHelper.getNodes()[0])];
        },

        _createDataSource: function (items, source, expandStates, level) {
            /// <summary>Overridable wrapper for populateDataSource</summary>
            this._populateDataSource(items, source, expandStates, level);
        },

        _populateDataSource: function (items, source, expandStates, level) {
            /// <summary>Creates source data for the given items.</summary>
            /// <param name="items" type="Object">The structure defining the tree for the grid.
            /// See CheckboxSelectionGrid function for details about gridItems format.</param>
            /// <param name="source" type="Array">Array of grid rows where every row is an array of values.</param>
            /// <param name="expandStates" type="Array">Array of numbers of the same size as 'source' argument 
            ///     containing number of children in the tree under every row recursively.</param>
            /// <param name="checkedItems" type="Object">The table allows for fast lookup of checked item IDs.</param>
            /// <param name="level" type="Number">Current level of the tree (1 is for the roots).</param>
            /// <returns>Returns number of given items including their children recursively.</returns>

            var itemIndex,
                totalItems = items.length,
                i, l;

            // Perform depth first walk of the tree while creating a row for every item in the tree and
            // calculating the number of children under every item recursively.
            for (i = 0, l = items.length; i < l; i += 1) {

                // Create a new row for the item
                itemIndex = source.length;

                // Store the data index on the item for reverse lookup purposes.
                items[i].dataIndex = itemIndex;

                source[itemIndex] = this._constructRow(items[i]);

                // Reserve the space for the number of items under this item
                expandStates[itemIndex] = 0;

                if (typeof items[i].children !== "undefined") {
                    expandStates[itemIndex] = this._populateDataSource(items[i].children, source, expandStates, level + 1);
                    totalItems += expandStates[itemIndex];
                }
            }

            return totalItems;
        },

        _constructRow: function (sourceRow) {
            /// <summary>Constructs an array of values from the source row which is used
            /// by the Checklist grid control to managed the items checked/unchecked.</summary>
            /// <param name="sourceRow">A row from the source data set.</param>

            // Prepare row values
            var itemSource = [].concat(sourceRow.values);

            // Add the ID (which is not stored as a value)
            itemSource[HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX] = sourceRow.id;

            return itemSource;
        },

        _addNewItem: function (node, parent) {
            /// <summary>Responds to a new item added to the data provider.</summary>
            /// <param name="node" type="Object">The node added to the data provider.</param>
            /// <param name="parent" type="Object">The parent the specified node was added to.</param>

            // Calculate the insertion position of the new node taking sort order
            // into consideration.
            this.refresh(true);
            
            var index = this.getDataIndexFromNode(node),
                parentIndex = this.getDataIndexFromNode(parent);

            // We need to add zero at the expanded state for the new item.
            if (index === 0) { // added to the root, so append to the end
                this._expandStates.splice(this._expandStates.length, 0, 0);
            }
            else {
                this._expandStates.splice(index, 0, 0);
            }

            // Update the expand states for the parents.
            this._updateExpandStates(parentIndex, 1);
        },
        
        _removeItem: function (node, parent, treeSize) {
            /// <summary>Remove a work item from the grid.</summary>
            /// <param name="node" type="Object">The node removed from the data provider.</param>
            /// <param name="parent" type="Object">The parent of the node removed.</param>
            /// <param name="treeSize" type="Number">The total number of children of the node removed (including the node itself).</param>
            
            Diag.assertParamIsObject(node, "node");
            Diag.assertParamIsObject(parent, "parent");
            Diag.assertParamIsNumber(treeSize, "treeSize");
                        
            var nodeIndex = this.getDataIndexFromNode(node),
                parentIndex = this.getDataIndexFromNode(parent);

            // Remove the expand states for the removed item and its children.
            this._expandStates.splice(nodeIndex, treeSize);

            // Update parents expand states
            this._updateExpandStates(parentIndex, -treeSize);
        
            // Update any indexes maintained using the data index
            this.refresh(true);
        },
        
        _updateItem: function (node) {
            /// <summary>Update a work item in the grid.</summary>
            /// <param name="node" type="Object">The edited node from the data provider.</param>
            
            Diag.assertParamIsObject(node, "node");
                        
            var oldNodeIndex = this.getDataIndexFromNode(node),
                newNodeIndex;
            
            // Recalculate the new position of the node in the tree
            this.refresh(true);
            
            // Move the expand states for the node and its children from the old location to the
            // new one.
            
            newNodeIndex = this.getDataIndexFromNode(node);
            
            this._moveExpandStatesForNode(oldNodeIndex, newNodeIndex);
        },

        _updateExpandStates: function (itemIndex, increment) {
            /// <summary>Updates the expand states to account for changes in the grid data.</summary>
            /// <param name="itemIndex" type="number">Index of the item to start updating at.</param>
            /// <param name="increment" type="integer">Number of items added or removed.  The expand states will be incremented by this value.</param>

            Diag.assertParamIsNumber(itemIndex, "itemIndex");
            Diag.assertParamIsNumber(increment, "increment");

            var nodeIndex = itemIndex,
                expandStates = this._expandStates;

            if (nodeIndex >= 0) {
                // Walk the expanded state tree and update the item and its parents.
                do {
                    if (expandStates[nodeIndex] < 0) { // if collapsed then decrement to add new element to the tree
                        expandStates[nodeIndex] -= increment;
                    }
                    else {
                        expandStates[nodeIndex] += increment;
                    }

                    nodeIndex = this.getParentNodeIndexForDataIndex(nodeIndex);
                } while (nodeIndex >= 0); // check if it has a parent
            }
        },
        
        _moveExpandStatesForNode: function (oldNodeIndex, newNodeIndex) {
            /// <summary>Moves the expand states for a node and all its children from oldNodeIndex to newNodeIndex.</summary>
            /// <param name="oldNodeIndex" type="Number">The source location of the node states to move.</param>
            /// <param name="newNodeIndex" type="Number">The destination location of the node states ot move.</param>
            
            var treeSize = Math.abs(this._expandStates[oldNodeIndex]) + 1,
                removedItems,
                i, l;
            
            // Pull the node out of its source location
            removedItems = this._expandStates.splice(oldNodeIndex, treeSize);
            
            // Push the node onto the array in the destination location
            for (l = removedItems.length, i = l - 1; i >= 0; i -= 1) {
                this._expandStates.splice(newNodeIndex, 0, removedItems[i]);
            }
        }
    };

    ///
    /// ChecklistDataAdapter - Provides hierarchical data to the ChecklistSelectionGrid control.
    ///

    ChecklistDataAdapter = function (fieldDataProvider, grid, options) {
        /// <summary>Description</summary>
        /// <param name="fieldDataProvider" type="object">field Data Provider</param>
        /// <param name="grid" type="object">grid</param>
        /// <param name="option" type="object">options that could include
        /// allEnabled: if all checkboxes are enabled or disabled
        /// rootNodeId: the root element to display checkboxes under
        /// noColumn: whether to add the column for checkboxes to the grid
        /// disabledTooltip: the tooltip text to show on disabled checkboxes
        /// </param>

        Diag.assertParamIsObject(fieldDataProvider, "fieldDataProvider");
        Diag.assertParamIsObject(grid, "grid");
        Diag.assertParamIsObject(options, "options");

        this._events = new Sys.EventHandlerList();
        this._checkboxRangeRootId = options.rootNodeId || null;
        this._noCheckboxes = options.noCheckboxes;

        // Default all checkboxes enabled to true
        this._allEnabled = options.allEnabled === undefined ? true :options.allEnabled;

        var that = this,
            columns = grid._options.columns,
            checkboxColumn = {
                canSortBy: false,
                text: "",
                width: 30,
                fixed: true,
                name: ChecklistDataAdapter.CHECK_COLUMN_NAME,
                getCellContents: function (rowInfo, dataIndex, expandedState, level, column, indentIndex, columnOrder) {
                    return that._createCheckboxCell(dataIndex, column);
                }
            };
        
        if (!options.noColumn) {
            columns.splice(ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX, 0, checkboxColumn);

            grid.getElement().keyup(function (e) {
                var dataIndex,
                    newState,
                    $checkBox;

                if (e.which === TFS.UI.KeyCode.SPACE) {
                    dataIndex = that._grid.getSelectedDataIndex();
                    $checkBox = that._findCheckbox(dataIndex);
                    if ($checkBox.length === 1 && !$checkBox.attr("disabled")) {
                        newState = !$checkBox.attr("checked");
                        $checkBox.attr("checked", newState);
                        that._setCheckedState(dataIndex, newState);
                    }
                }   
            });
        }
        this._disabledTooltip = options.disabledTooltip || "";
        if (options.disableChildren && $.isFunction(options.disableChildren)) {
            this._disableChildren = options.disableChildren;
        }
        else {
            this._disableChildren = function () {
                return true;
            };
        }

        this.baseConstructor(fieldDataProvider, grid);
        this.initialize(options.checkedItemIds || []);        
    };

    ChecklistDataAdapter.extend({
        _CHECKBOX_COLUMN_INDEX: 0,
        _LABEL_COLUMN_INDEX: 1,
        _CHECK_CHANGED: "checked-items-changed",
        CHECK_COLUMN_NAME: "checks-column"
    });

    ChecklistDataAdapter.inherit(HierarchicalGridDataAdapter, {
        _checkedItems: null,
        _itemStates: null,
        _events: null,
        _disabledTooltip: null,
        _checkboxRangeRootId: null,
        _checkboxRangeBegin: null,
        _checkboxRangeEnd: null,
        _allEnabled: true,
        _blockedCheckIds: null, // List of grid row id's that should not allow checkbox selection
        _disableChildren: null, // a callback that allows external control of whether to disable children on check or not
        _noCheckboxes: null, // indicates that no checkboxes should appear on the grid yet we still want the data adapter since this is temporary

        initialize: function (checkedItemIds) {
            /// <summary>Initializes the data provider and prepares it for use
            /// by the checklist selection grid. </summary>
            /// <param name="checkedItemIds">A collection of item IDs representing the checked
            /// items.</param>

            var checkedItems = {},
                i, l;

            // Populate lookup table with checked item IDs
            for (i = 0, l = checkedItemIds.length; i < l; i += 1) {
                checkedItems[checkedItemIds[i]] = checkedItemIds[i];
            }

            this._checkedItems = checkedItems;
            this._blockedCheckIds = {};
        },

        hasCheckBox: function (dataIndex) {
            /// <summary>Determine if a row has a checkbox</summary>
            /// <param name="dataIndex" type="Number">index of the row to check</param>
            /// <returns type="Boolean" />
            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            return dataIndex >= this._checkboxRangeBegin && dataIndex <= this._checkboxRangeEnd;
        },

        updateCheckboxesRange: function (expandStates){
            /// <summary>Updates the checkbox range</summary>
            /// <param name="expandStates" type="Array">The expand states</param>
            Diag.assertParamIsArray(expandStates, "expandStates", false);

            var node,
                treeSize,
                rootIndex;

            if (this._checkboxRangeRootId) {
                node = this.dataProvider.getNodeFromId(this._checkboxRangeRootId);

                if (node) {
                    rootIndex = this.getDataIndexFromNode(node);
                    treeSize = Math.abs(expandStates[rootIndex]);
                    this._checkboxRangeBegin = node.dataIndex + 1;
                    this._checkboxRangeEnd = node.dataIndex + treeSize;
                }
            }
            else if (this._noCheckboxes) {
                this._checkboxRangeBegin = -1;
                this._checkboxRangeEnd = -1;
            }
            else {
                this._checkboxRangeBegin = 0;
                this._checkboxRangeEnd = this._flattenedItems.length - 1;
            }
        },

        isLeafNode: function (node) {
            /// <summary>Determines whether the node is a leaf node</summary>
            /// <param name="node" type="Object">A tree node</param>
            /// <returns type="Boolean" />
            Diag.assertParamIsObject(node, "node");

            return this._expandStates[node.dataIndex] === 0;
        },

        blockCheck: function (id) {
            /// <summary>Disables and blocks the checking operation for the provided data index</summary>
            /// <param name="id" type="String">The id the row in the grid</param>
            Diag.assertParamIsString(id, "id");

            this._blockedCheckIds[id] = true;
            this._grid.updateRow(undefined, this._getDataIndexFromId(id));
        },

        unblockCheck: function (id) {
            /// <summary>Ensures enablement of the checking operation for the provided data index</summary>
            /// <param name="id" type="String">The id of the row in the grid</param>
            Diag.assertParamIsString(id, "id");

            if (this._blockedCheckIds.hasOwnProperty(id)) {
                delete this._blockedCheckIds[id];
                this._grid.updateRow(undefined, this._getDataIndexFromId(id));
            }
        },

        setCheckboxRangeRoot: function (id) {
            /// <summary>Sets the root of the check box range</summary>
            /// <param name="id" type="String">Id of the node to be the root of check boxes</param>
            Diag.assertParamIsString(id, "id");

            this._checkboxRangeRootId = id;
            this.refresh();
        },

        getCheckboxRangeRoot: function (){
            /// <summary>Gets the currently selected check boxes root id</summary>
            /// <returns type="Number">The id of the checkbox range root</returns>
            return this._checkboxRangeRootId;
        },

        _getCheckboxCellId: function (dataIndex) {
            /// <summary>Gets the value used for the ID attribute of the checkbox DOM element at a given index</summary>
            /// <param name="dataIndex" type="number">data index of the row</param>
            /// <returns type="String">A (unique) id for the checkbox</returns>
            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            var gridId = this._grid._getId() || "";

            return String.format("checkbox-{0}-{1}", gridId, dataIndex);
        },

        _findCheckbox: function (dataIndex) {
            /// <summary>Attempts to find the checkbox associated with a given dataIndex</summary>
            /// <param name="dataIndex" type="number">data index of the row</param>
            /// <returns type="jQuery">A jQuery object containing the checkbox for the given dataIndex (or an empty jQuery object if one doesn't exist.</returns>
            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            var checkBoxId = this._getCheckboxCellId(dataIndex),
                $checkBox = $("#" + checkBoxId);

            Diag.assert($checkBox.length <= 1, "Expected there to be at most one checkbox with the given ID: " + checkBoxId + ", instead saw: " + $checkBox.length);
            return $checkBox;
        },

        _createCheckboxCell: function (dataIndex, column) {
            /// <summary>create checkbox cell at specific row in a column</summary>
            /// <param name="dataIndex" type="number">index of the row</param>
            /// <param name="column" type="object">column object</param>
            /// <returns type="jQuery">The checkbox cell</returns>
            Diag.assertParamIsNumber(dataIndex, "dataIndex");
            Diag.assertParamIsObject(column, "column");
            Diag.assert(ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX === column.index,
                String.format("Expected that the checkbox column is in position {0}. Actual index is {1}", ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX, column.index));

            var $cell,
                checkboxId = this._getCheckboxCellId(dataIndex),
                $checkbox,
                $label,
                enabled,
                checked;
            
            // Create the cell which will contain the checkbox
            $cell = $(domElem("div", "grid-cell"));
            $cell.width(column.width || 20);                   

            // Get the current checkbox state
            if (this.hasCheckBox(dataIndex)) {
                enabled = this.getItemEnabled(dataIndex);
                checked = this.getItemChecked(dataIndex);

                $label = $("<label class='hidden' for='" + checkboxId + "'/>")
                    .text(this._grid.getColumnValue(dataIndex, ChecklistDataAdapter._LABEL_COLUMN_INDEX));

                $cell.append($label);

                // Create the checkbox element and set its state
                $checkbox = $("<input type='checkbox' tabindex='-1' id='" + checkboxId + "'>")
                    .data("checkbox-data-index", dataIndex)                    
                    .attr('checked', checked)
                    .attr("disabled", !enabled);

                this._setCheckboxDefaultTitle($checkbox);
                $checkbox.click(delegate(this, this._onCheckboxClicked));
                $cell.append($checkbox);
            }

            return $cell;
        },        
       
        _onCheckboxClicked: function (e) {
            /// <summary>The handler is invoked when a checkbox on a grid row is clicked.</summary>
            /// <param name="e" type="Object">jQuery event object.</param>

            TFS.Diag.logTracePoint('TFS.UI.Controls.Data.ChecklistDataAdapter._onCheckboxClicked.start');

            var $checkbox = $(e.currentTarget),
                dataIndex = $checkbox.data("checkbox-data-index"),
                checked = $checkbox.is(":checked");

            this._setCheckedState(dataIndex, checked);
        },

        _setCheckedState: function (dataIndex, checked){

            // Remember the new state of this cell
            this.setCheckboxStateData(dataIndex, checked);

            // Don't return anything to allow the default action for the event execute
            this._raiseCheckedItemsChanged({ id: this._getIdFromDataIndex(dataIndex) });
        },

        _raiseCheckedItemsChanged: function (args) {
            /// <summary>Notifies listeners of that a work item was removed.</summary>
            /// <param name="args" type="object">args</param>
            Diag.assertParamIsObject(args, "args");

            var handler = this._events.getHandler(ChecklistDataAdapter._CHECK_CHANGED);

            if (handler) {
                handler(this, args);
            }
        },

        attachCheckedItemsChanged: function (handler) {
            /// <summary>Attach a handler for the removed item event. </summary>
            /// <param name="handler" type="Function">function</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.addHandler(ChecklistDataAdapter._CHECK_CHANGED, handler);
        },

        detachCheckedItemsChanged: function (handler) {
            /// <summary>Remove a handler for the removed item event.</summary>
            /// <param name="handler" type="Function">The handler to remove</param>
            Diag.assertParamIsFunction(handler, "handler");

            this._events.removeHandler(ChecklistDataAdapter._CHECK_CHANGED, handler);
        },

        _createDataSource: function (items, source, expandStates, level) {
            /// <summary>OVERRIDE: create the datasource for the grid</summary>

            var i, l,
                itemStates = [],
                dataSource,
                expandState,
                columnIndex = ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX,
                idIndex = HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX,
                currentIndex;

            this._base(items, source, expandStates, level);

            dataSource = this._flattenedItems;
            this.updateCheckboxesRange(expandStates);

            for (i = 0, l = expandStates.length; i < l; i++) {
                if (this.hasCheckBox(i)) {
                    expandState = Math.abs(expandStates[i]);

                    itemStates[i] = true;

                    if (this._disableChildren(dataSource[i][idIndex]) && dataSource[i][columnIndex]) {
                        currentIndex = i;

                        while (i < currentIndex + expandState) {
                            itemStates[++i] = false;
                            dataSource[i][columnIndex] = false;
                            delete this._checkedItems[dataSource[i][idIndex]];
                        }
                    }
                }
                else {
                    delete this._checkedItems[dataSource[i][idIndex]];
                    dataSource[i][columnIndex] = false;
                }
            }

            this._itemStates = itemStates;           
        },

        setItemState: function (id, enabled) {
            /// <summary>Sets the enabled state of the row</summary>
            /// <param name="id" type="String">The item ID used to look up the item in the state cache.</param>
            /// <param name="enabled" type="Boolean">The new state of the row.</param>
            Diag.assertIsNotNull(id, "id");
            Diag.assertParamIsBool(enabled, "enabled");

            if (enabled) {
                this._checkedItems[id] = id;
            }
            else {
                delete this._checkedItems[id];
            }
        },

        getItemChecked: function (dataIndex){
            /// <summary>Return Whether the item at the dataIndex is checked</summary>
            /// <param name="dataIndex" type="Number">index of item to check if checked</param>
            /// <returns type="boolean" />

            return this._grid.getColumnValue(dataIndex, ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX);
        },

        setItemTitle: function (id, title) {
            /// <summary>Set the title of the checkbox identified by a given item id</summary>
            /// <param name="id" type="String">The item ID used to look up the item in the state cache.</param>
            /// <param name="title" type="String">The title to set.</param>
            Diag.assertParamIsString(id, "id");
            Diag.assertParamIsString(title, "title");

            var dataIndex = this._getDataIndexFromId(id),
                $checkbox = this._findCheckbox(dataIndex);
            
            $checkbox.attr("title", title);
        },

        resetItemTitle: function (id) {
            /// <summary>Reset the title of the checkbox identified by a given item id</summary>
            /// <param name="id" type="String">The item ID used to look up the item in the state cache.</param>
            Diag.assertParamIsString(id, "id");

            var dataIndex = this._getDataIndexFromId(id);
                $checkbox = this._findCheckbox(dataIndex); 
            
            this._setCheckboxDefaultTitle($checkbox);
        },

        _setCheckboxDefaultTitle: function ($checkbox) {
            /// <summary>Sets the title of the checkbox to the default value</summary>
            /// <param name="$checkbox" type="jQuery">The jQuery object for the checkbox</param>
            Diag.assertParamIsJQueryObject($checkbox, "$checkbox");
            Diag.assert($checkbox.length <= 1, "Expected at most one checkbox to be passed in. Got: " + $checkbox.length);

            $checkbox.attr("title", $checkbox.attr("disabled") ? this._disabledTooltip : "");
        },

        getCheckedItemIds: function () {
            /// <summary>Allows accessing the list of grid items that are currently checked.</summary>
            /// <returns>Returns array of checked item ids.</returns>
            var i, l,
                result = [],
                columnIndex = ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX,
                idIndex = HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX,
                checkedItems = {},
                dataSource = this._flattenedItems,
                itemStates = this._itemStates,
                value;

            // clean up checked items
            for (i = 0, l = itemStates.length; i < l; i++) {
                if (itemStates[i] && dataSource[i][columnIndex]) {
                    checkedItems[dataSource[i][idIndex]] = dataSource[i][idIndex];
                }
            }

            this._checkedItems = checkedItems;

            for (value in checkedItems) {
                if (checkedItems.hasOwnProperty(value)) {
                    result.push(checkedItems[value]);
                }
            }

            return result;
        },

        setCheckboxStateData: function (dataIndex, state) {
            /// <summary>Updates checkbox related data for grid row with the new state (without touching the actual checkbox element).</summary>
            /// <param name="dataIndex" type="Number">The row index.</param>
            /// <param name="state" type="Boolean">New state for the row's checkbox.</param>
            Diag.assert(typeof dataIndex === "number", "Expected to be a number");
            Diag.assert(typeof state === "boolean", "Expected to be a number");

            var i,
                children = Math.abs(this._expandStates[dataIndex]),
                dataSource = this._grid._dataSource,
                id = dataSource[dataIndex][HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX];

            dataSource[dataIndex][ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX] = state;
            if (this._disableChildren(id)) {
                for (i = dataIndex + 1; i <= dataIndex + children; i++) {
                    dataSource[i][ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX] = false;
                    this.setItemState(dataSource[i][HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX], false);
                    this._itemStates[i] = !state;
                    this._grid.updateRow(undefined, i);
                }
            }

            // Update the checked items table
            this.setItemState(id, state);
            this._grid.updateRow(undefined, dataIndex);
        },

        getCheckboxState: function (dataIndex) {
            /// <summary>Gets the checkbox state for the provided data index.</summary>
            /// <param name="dataIndex" type="Number">The data index to get the checkbox state for.</param>
            /// <returns type="Boolean">True when the checkbox is checked and false otherwise.</returns>
            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            var dataSource = this._grid._dataSource;

            if (dataIndex >= 0) {
                return dataSource[dataIndex][ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX];
            }

            return false;
        },

        getItemEnabled: function (dataIndex) {
            /// <summary>Determine whether the checkbox at the specified dataIndex is enabled</summary>
            /// <returns type="Boolean" />
            return this._allEnabled && this._itemStates[dataIndex] && !this._blockedCheckIds[this._getIdFromDataIndex(dataIndex)];
        },

        _getIdFromDataIndex: function (dataIndex) {
            /// <summary>Get the id of the row at the specified dataIndex</summary>
            /// <returns type="String" />
            Diag.assertParamIsNumber(dataIndex, "dataIndex");

            return this._flattenedItems[dataIndex][HierarchicalGridDataAdapter._ITEM_ID_DATA_SOURCE_INDEX];
        },

        _getDataIndexFromId: function (id) {
            /// <summary>Get the dataIndex of the row for the specified item id</summary>
            /// <param name="id" type="String">The item ID used to look up the item in the state cache.</param>
            /// <returns type="Number" />
            Diag.assertParamIsString(id, "id");

            var node = this.dataProvider.getNodeFromId(id);
            Diag.assert(Boolean(node), "Expected to find a node for id: " + id);

            return (node ? node.dataIndex : undefined);
        },

        getBranchCheckedState: function () {
            /// <summary>Gets the branch-level checked state based on the state grid items.</summary>
            /// <returns type="Boolean" />
            var state = true,
                source = this._grid._dataSource,
                rootNode = this.dataProvider.getRootNode(),
                roots = [],
                i, l;

            if (rootNode) {
                roots = rootNode.children;
            }

            // Do not check the header checkbox if there are no items.
            if (!roots || roots.length === 0) {
                state = false;
            }
            else {
                // The header checkbox can be checked only when all root items are checked.
                for (i = 0, l = roots.length; i < l && state; i += 1) {
                    state = source[roots[i].dataIndex][ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX];
                }
            }

            return state;
        },

        _constructRow: function (sourceRow) {
            /// <summary>OVERRIDE: Constructs an array of values from the source row which is used
            /// by the Checklist grid control to managed the items checked/unchecked.</summary>
            /// <param name="sourceRow">A row from the source data set.</param>
            Diag.assertParamIsObject(sourceRow, "sourceRow");

            var newRow = this._base(sourceRow),
                checkboxState = this._checkedItems.hasOwnProperty(sourceRow.id);

            newRow.splice(ChecklistDataAdapter._CHECKBOX_COLUMN_INDEX, 0, checkboxState);

            return newRow;
        }
    });

    ///
    /// TabDelimitedTableFormatter - Provides a formatter object that formats selected rows in a grid into a plain-text tab-delimited table.
    ///
    
    TabDelimitedTableFormatter = function(grid, options) {
        
        Diag.assertParamIsObject(grid, "grid");
        
        this._options = options;
        this._grid = grid;
    };
    
    TabDelimitedTableFormatter.prototype = {
        _options: null,
        _grid: null,
        
        getTableFromSelectedItems: function() {
            /// <summary>Iterates through the selected rows and builds a table containing the results.</summary>
            /// <returns>A tab-delimited plain-text table containing all rows and all columns in the current selection.</returns>
            
            TFS.Diag.logTracePoint('TFS.UI.Controls.Data.TabDelimitedTableFormatter.getTableFromSelectedItems.start');
            
            var grid = this._grid,
                selectedItems = grid.getSelectedDataIndices(),
                sb = new Core.StringBuilder(),
                value,
				columns;
                
			columns = grid.getColumns();
            
			if ((selectedItems.length === 0) || (columns.length === 0)) {
                return "";
            }
            
            // Construct the column headers
            $.each(columns, function (index, column) {
                if (index > 0) {
                    sb.append(Core.StringConstants.tab);
                }
                
                sb.append(column.text);
            });
            
            sb.appendNewLine();
            
            $.each(selectedItems, function (index, row) {
                if (index > 0) {
                    sb.appendNewLine();
                }
                
				sb.append($.map(columns, function (column) {
            		return grid.getColumnText(row, column);
            	}).join(Core.StringConstants.tab));
            });

            TFS.Diag.logTracePoint('TFS.UI.Controls.Data.TabDelimitedTableFormatter.getTableFromSelectedItems.complete');
            
            return sb.toString();
        }
    };
        
    return {
        FieldDataProvider: FieldDataProvider,
        HierarchicalGridDataAdapter: HierarchicalGridDataAdapter,
        ChecklistDataAdapter: ChecklistDataAdapter,
        TabDelimitedTableFormatter: TabDelimitedTableFormatter
    };
});


// SIG // Begin signature block
// SIG // MIIawwYJKoZIhvcNAQcCoIIatDCCGrACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFLVLxsGyLR4f
// SIG // t56SshGPAPgUrfleoIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AAArOTJIwbLJSPMAAAAAACswDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTEyMDkwNDIx
// SIG // MTIzNFoXDTEzMTIwNDIxMTIzNFowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAKa2MA4DZa5QWoZrhZ9IoR7JwO5eSQeF4HCWfL65
// SIG // X2JfBibTizm7GCKlLpKt2EuIOhqvm4OuyF45jMIyexZ4
// SIG // 7Tc4OvFi+2iCAmjs67tAirH+oSw2YmBwOWBiDvvGGDhv
// SIG // sJLWQA2Apg14izZrhoomFxj/sOtNurspE+ZcSI5wRjYm
// SIG // /jQ1qzTh99rYXOqZfTG3TR9X63zWlQ1mDB4OMhc+LNWA
// SIG // oc7r95iRAtzBX/04gPg5f11kyjdcO1FbXYVfzh4c+zS+
// SIG // X+UoVXBUnLjsfABVRlsomChWTOHxugkZloFIKjDI9zMg
// SIG // bOdpw7PUw07PMB431JhS1KkjRbKuXEFJT7RiaJMCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBSlGDNTP5VgoUMW747G
// SIG // r9Irup5Y0DAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQB+zLB75S++
// SIG // 51a1z3PbqlLRFjnGtM361/4eZbXnSPObRogFZmomhl7+
// SIG // h1jcxmOOOID0CEZ8K3OxDr9BqsvHqpSkN/BkOeHF1fnO
// SIG // B86r5CXwaa7URuL+ZjI815fFMiH67holoF4MQiwRMzqC
// SIG // g/3tHbO+zpGkkSVxuatysJ6v5M8AYolwqbhKUIzuLyJk
// SIG // pajmTWuVLBx57KejMdqQYJCkbv6TAg0/LCQNxmomgVGD
// SIG // ShC7dWNEqmkIxgPr4s8L7VY67O9ypwoM9ADTIrivInKz
// SIG // 58ScCyiggMrj4dc5ZjDnRhcY5/qC+lkLeryoDf4c/wOL
// SIG // Y7JNEgIjTy2zhYQ74qFH6M8VMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAx
// SIG // MjQyMjMzMzlaFw0xNDA0MjQyMjMzMzlaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34
// SIG // KHy8BXt/re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SL
// SIG // qT+oN+EtUO01Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Un
// SIG // m/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEV
// SIG // XAhlHSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpO
// SIG // oPXJCiHiquMBNkf9L4JqgQP1qTXclFed+0vUDoLbOI8S
// SIG // /uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
// SIG // ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fF
// SIG // SsjoJ4TTfsZDC5UABbFPZXHFAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGm
// SIG // WjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRm
// SIG // YWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMwNTIzNDRh
// SIG // ZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEAMdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8
// SIG // dmiekKGcC8N/oozxTbqVOfsN4OGb9F0kDxuNiBU6fNut
// SIG // zrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
// SIG // sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2y
// SIG // RPnwPJNtQtjofOYXoJtoaAko+QKX7xEDumdSrcHps3Om
// SIG // 0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUoXgSp
// SIG // Rm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD3
// SIG // 7zTQYx2xQmdKDku/Og7vtpU6pzjkJZIIpohmgjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggStMIIEqQIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAALARrwqL0Duf3QAB
// SIG // AAAAsDAJBgUrDgMCGgUAoIHGMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBSL+lFqGmQ8
// SIG // dFZirK0m2sdMoiHK1jBmBgorBgEEAYI3AgEMMVgwVqA8
// SIG // gDoAVABGAFMALgBVAEkALgBDAG8AbgB0AHIAbwBsAHMA
// SIG // LgBEAGEAdABhAC4AZABlAGIAdQBnAC4AagBzoRaAFGh0
// SIG // dHA6Ly9taWNyb3NvZnQuY29tMA0GCSqGSIb3DQEBAQUA
// SIG // BIIBAF7loQRYN5HamHrjMDY2MkZKp/0XMXURH2o4c61H
// SIG // /FkHsk98hX9+Me/HV4hbrdmh7jpljNtmBMEPIAs+2iou
// SIG // oPZamozmGeynfNyHb5MVcB0fI+YEXZIlgeRN90uw7QKV
// SIG // A4vFRDTa9jyWbC+yo2bC2dZKl1WOhKoWSsHiMK1/Rj4u
// SIG // 9O3LlnL9kfpvqMR7tTSAVheBG9r8wOv9ZiD1rbT4gZI8
// SIG // 8ox6W6Nsw/02Vzy1rV8ZqnfCwvq+78XZGSe5X2cFkcHF
// SIG // GMIIymVtSSIaS2zAeEE0pVgob6ER8DEv/T4oyieH6z3A
// SIG // LwVBmTuAKDH8DV0R0U7V+nfdmtZKYbysY9c3IamhggIo
// SIG // MIICJAYJKoZIhvcNAQkGMYICFTCCAhECAQEwgY4wdzEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgUENBAhMzAAAAKzkySMGyyUjz
// SIG // AAAAAAArMAkGBSsOAwIaBQCgXTAYBgkqhkiG9w0BCQMx
// SIG // CwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0xMzAz
// SIG // MTUwNjMzNTVaMCMGCSqGSIb3DQEJBDEWBBRWOY8D0uLy
// SIG // OqteoK13kwGdN6kmQjANBgkqhkiG9w0BAQUFAASCAQAb
// SIG // 6Peg3tYWECmVVV4qh4stH4RvB2vt1XHZn4B3ZLmSDxCj
// SIG // gXC+uOaTgDl48te3jpfLUJxx/DOBBfAWfZ9LsPJO93P3
// SIG // Mf/XBjDyE05zsTp4+SBOstyuDm4COQGj9bXwOvuEu4EP
// SIG // c06UqeiVsVR1v2tfpWb83XfkyiVu0HmaEvZ5Bt9bm1nQ
// SIG // PkhveQlzX0I/bHqEDGc9o4G191RVB4oU2VQSqrPZvl0m
// SIG // bvyR6Wehq3GTghgZwbNPZivz5wLfd4aLX4hKURHE+8EA
// SIG // Xiim9IoqMtpEBBhU7NHN86zz9W2/i5Y90jD05/YAEa/X
// SIG // TSuY15hPklxg3Y3tsGidk/MghYv/gc5F
// SIG // End signature block
